@extends('admin.layout')

@section('content')
@if(session('success'))
<div class="alert alert-success" style="background: var(--primary-soft); color: var(--primary); padding: 12px 16px; border-radius: 8px; margin-bottom: 16px; border: 1px solid var(--primary);">
    {{ session('success') }}
</div>
@endif

<div class="panel-card">
    <div class="panel-header">
        <h2>{{ __('admin.users.title') }}</h2>
        <form method="GET" action="{{ route('admin.users') }}" class="search-form">
            <input type="text" name="search" placeholder="{{ __('admin.users.search_placeholder') }}" value="{{ request('search') }}">
            <select name="status">
                <option value="">{{ __('admin.users.all_status') }}</option>
                <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>{{ __('admin.users.active') }}</option>
                <option value="inactive" {{ request('status') === 'inactive' ? 'selected' : '' }}>{{ __('admin.users.inactive') }}</option>
            </select>
            <button type="submit">{{ __('admin.users.search') }}</button>
        </form>
    </div>

    <div class="table-container">
        <table class="data-table">
            <thead>
                    <tr>
                        <th>{{ __('admin.table.id') }}</th>
                        <th>{{ __('admin.users.wallet_address') }}</th>
                        <th>{{ __('admin.users.name') }}</th>
                        <th>{{ __('admin.users.score') }}</th>
                        <th>{{ __('admin.users.balance') }}</th>
                        <th>{{ __('admin.users.status') }}</th>
                        <th>{{ __('admin.users.verified') }}</th>
                        <th>{{ __('admin.table.actions') }}</th>
                    </tr>
            </thead>
            <tbody>
                @forelse($users as $user)
                <tr>
                    <td>{{ str_pad($user->id, 6, '0', STR_PAD_LEFT) }}</td>
                    <td>{{ substr($user->wallet_address, 0, 10) }}...</td>
                    <td>{{ $user->name }}</td>
                    <td>
                        <span style="display: flex; align-items: center; gap: 4px;">
                            <span>⭐</span>
                            <span>{{ number_format($user->account_weight_score, 2) }}</span>
                        </span>
                    </td>
                    <td title="Total Portfolio (USD) – matches Assets page">${{ number_format($totalPortfolioByUser[$user->id] ?? 0, 2) }}</td>
                    <td>
                        <span class="badge {{ $user->is_active ? 'badge-success' : 'badge-danger' }}">
                            {{ $user->is_active ? __('admin.users.active') : __('admin.users.inactive') }}
                        </span>
                    </td>
                    <td>
                        <span class="badge {{ $user->is_verified ? 'badge-success' : 'badge-warning' }}">
                            {{ $user->is_verified ? __('admin.users.verified') : __('admin.users.pending') }}
                        </span>
                    </td>
                    <td>
                        <div style="display: flex; gap: 6px; flex-wrap: wrap;">
                            <a href="{{ route('admin.users.show', $user->id) }}" class="btn btn-sm btn-primary">{{ __('admin.users.view') }}</a>
                            @if(!$user->is_verified)
                            <button type="button" class="btn btn-sm btn-success" title="{{ __('admin.users.verify_user') }}" onclick="showVerifyModal({{ $user->id }}, '{{ $user->name }}')">✓ {{ __('admin.users.verify_user') }}</button>
                            @endif
                            <button type="button" class="btn btn-sm btn-warning" title="{{ __('admin.users.set_reset_pin') }}" onclick="showPinModal({{ $user->id }}, '{{ addslashes($user->name) }}', {{ $user->withdrawal_pin_hash ? 'true' : 'false' }})">{{ __('admin.users.set_reset_pin') }}</button>
                            <button type="button" class="btn btn-sm btn-danger" title="{{ __('admin.users.delete_user') }}" onclick="showDeleteModal({{ $user->id }}, {{ json_encode($user->name) }})">{{ __('admin.common.delete') }}</button>
                        </div>
                    </td>
                </tr>
                @empty
                <tr>
                    <td colspan="8" class="text-center">{{ __('admin.users.no_found') }}</td>
                </tr>
                @endforelse
            </tbody>
        </table>
    </div>

    @include('admin.partials.pagination', ['items' => $users])
</div>

<!-- Delete User Confirmation Modal (custom toast-style confirm) -->
<div id="deleteUserModal" class="modal-backdrop" style="display: none;">
    <div class="modal-card">
        <div class="modal-header">
            <h3 class="modal-title">{{ __('admin.users.delete_user_modal_title') }}</h3>
        </div>
        <div class="modal-body">
            <p style="color: var(--text); margin: 0;">{{ __('admin.users.delete_user_confirm_text') }} <strong id="deleteUserName"></strong>? {{ __('admin.users.delete_user_cannot_undo') }}</p>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-outline" onclick="closeDeleteModal()">{{ __('admin.common.cancel') }}</button>
            <form id="deleteUserForm" method="POST" action="#" style="display: inline;">
                @csrf
                @method('DELETE')
                <button type="submit" class="btn btn-danger">{{ __('admin.users.confirm_delete') }}</button>
            </form>
        </div>
    </div>
</div>

<!-- Verify Confirmation Modal -->
<div id="verifyModal" class="modal-backdrop" style="display: none;">
    <div class="modal-card">
        <div class="modal-header">
            <h3 class="modal-title">{{ __('admin.users.verify_modal_title') }}</h3>
        </div>
        <div class="modal-body">
            <p style="color: var(--text); margin: 0;">{{ __('admin.users.verify_confirm_text') }} <strong id="verifyUserName"></strong>?</p>
            <p style="color: var(--muted); margin: 8px 0 0 0; font-size: 14px;">{{ __('admin.users.verify_kyc_note') }}</p>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-outline" onclick="closeVerifyModal()">{{ __('admin.common.cancel') }}</button>
            <form id="verifyForm" method="POST" style="display: inline;">
                @csrf
                <button type="submit" class="btn btn-success">{{ __('admin.users.confirm_verify') }}</button>
            </form>
        </div>
    </div>
</div>

<script>
let currentVerifyUserId = null;

function showDeleteModal(userId, userName) {
    document.getElementById('deleteUserName').textContent = userName;
    document.getElementById('deleteUserForm').action = '{{ route('admin.users.destroy', ':id') }}'.replace(':id', userId);
    document.getElementById('deleteUserModal').style.display = 'flex';
}

function closeDeleteModal() {
    document.getElementById('deleteUserModal').style.display = 'none';
}

document.getElementById('deleteUserModal').addEventListener('click', function(e) {
    if (e.target === this) closeDeleteModal();
});
document.getElementById('deleteUserModal').addEventListener('keydown', function(e) {
    if (e.key === 'Escape') closeDeleteModal();
});

function showVerifyModal(userId, userName) {
    currentVerifyUserId = userId;
    document.getElementById('verifyUserName').textContent = userName;
    document.getElementById('verifyForm').action = '{{ route('admin.users.verify', ':id') }}'.replace(':id', userId);
    document.getElementById('verifyModal').style.display = 'flex';
}

function closeVerifyModal() {
    document.getElementById('verifyModal').style.display = 'none';
    currentVerifyUserId = null;
}

// Close modal on backdrop click
document.getElementById('verifyModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeVerifyModal();
    }
});

// Close modal on ESC key
document.getElementById('verifyModal').addEventListener('keydown', function(e) {
    if (e.key === 'Escape') closeVerifyModal();
});

// --- Set/Reset PIN Modal (in-app, no browser confirm) ---
let currentPinUserId = null;
let currentPinUserHasPin = false;

function showPinModal(userId, userName, userHasPin) {
    currentPinUserId = userId;
    currentPinUserHasPin = userHasPin;
    document.getElementById('pinModalUserName').textContent = userName;
    document.getElementById('pinNewPin').value = '';
    document.getElementById('pinConfirm').value = '';
    document.getElementById('pinError').textContent = '';
    document.getElementById('pinError').style.display = 'none';
    document.getElementById('pinClearBtn').style.display = userHasPin ? 'inline-block' : 'none';
    document.getElementById('pinModal').style.display = 'flex';
}

function closePinModal() {
    document.getElementById('pinModal').style.display = 'none';
    currentPinUserId = null;
}

function submitSetPin() {
    const newPin = (document.getElementById('pinNewPin').value || '').replace(/\D/g, '');
    const confirmPin = (document.getElementById('pinConfirm').value || '').replace(/\D/g, '');
    const errEl = document.getElementById('pinError');
    if (newPin.length !== 6) {
        errEl.textContent = @json(__('admin.users.pin_must_6'));
        errEl.style.display = 'block';
        return;
    }
    if (newPin !== confirmPin) {
        errEl.textContent = @json(__('admin.users.pin_no_match'));
        errEl.style.display = 'block';
        return;
    }
    errEl.style.display = 'none';
    const form = document.getElementById('pinSetForm');
    form.action = '{{ route('admin.users.set-pin', ':id') }}'.replace(':id', currentPinUserId);
    form.pin.value = newPin;
    form.pin_confirmation.value = confirmPin;
    form.submit();
}

function submitClearPin() {
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = '{{ route('admin.users.reset-pin', ':id') }}'.replace(':id', currentPinUserId);
    const csrf = document.createElement('input');
    csrf.type = 'hidden';
    csrf.name = '_token';
    csrf.value = '{{ csrf_token() }}';
    form.appendChild(csrf);
    document.body.appendChild(form);
    form.submit();
}
</script>

<!-- Set/Reset PIN Modal -->
<div id="pinModal" class="modal-backdrop" style="display: none;">
    <div class="modal-card">
        <div class="modal-header">
            <h3 class="modal-title">{{ __('admin.users.pin_modal_title') }}</h3>
        </div>
        <div class="modal-body">
            <p style="color: var(--text); margin: 0 0 16px 0;">{{ __('admin.users.pin_modal_user') }} <strong id="pinModalUserName"></strong></p>
            <p style="color: var(--muted); font-size: 14px; margin-bottom: 16px;">{{ __('admin.users.pin_modal_note') }}</p>
            <form id="pinSetForm" method="POST" action="#" style="display: block;">
                @csrf
                <input type="hidden" name="pin" id="pinSetFormPin" value="">
                <input type="hidden" name="pin_confirmation" id="pinSetFormConfirm" value="">
                <div class="form-group" style="margin-bottom: 12px;">
                    <label style="display: block; margin-bottom: 4px; color: var(--text);">{{ __('admin.users.new_pin') }}</label>
                    <input type="password" id="pinNewPin" inputmode="numeric" maxlength="6" placeholder="••••••" class="form-control" style="width: 100%; padding: 8px 12px; border-radius: 8px; border: 1px solid var(--border); background: var(--bg); color: var(--text);">
                </div>
                <div class="form-group" style="margin-bottom: 12px;">
                    <label style="display: block; margin-bottom: 4px; color: var(--text);">{{ __('admin.users.confirm_new_pin') }}</label>
                    <input type="password" id="pinConfirm" inputmode="numeric" maxlength="6" placeholder="••••••" class="form-control" style="width: 100%; padding: 8px 12px; border-radius: 8px; border: 1px solid var(--border); background: var(--bg); color: var(--text);">
                </div>
                <div id="pinError" style="display: none; color: #dc3545; font-size: 14px; margin-bottom: 12px;"></div>
            </form>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-outline" onclick="closePinModal()">{{ __('admin.common.cancel') }}</button>
            <button type="button" id="pinClearBtn" class="btn btn-outline" style="display: none;" onclick="submitClearPin()">{{ __('admin.users.clear_pin') }}</button>
            <button type="button" class="btn btn-warning" onclick="submitSetPin()">{{ __('admin.users.set_pin_btn') }}</button>
        </div>
    </div>
</div>

<script>
document.getElementById('pinModal').addEventListener('click', function(e) {
    if (e.target === this) closePinModal();
});
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape' && document.getElementById('pinModal').style.display === 'flex') closePinModal();
});
// Form submit: set hidden fields from visible inputs before submit
document.getElementById('pinSetForm').addEventListener('submit', function() {
    this.querySelector('input[name="pin"]').value = (document.getElementById('pinNewPin').value || '').replace(/\D/g, '');
    this.querySelector('input[name="pin_confirmation"]').value = (document.getElementById('pinConfirm').value || '').replace(/\D/g, '');
});
</script>
@endsection

