@extends('admin.layout')

@section('content')
@if(session('success'))
<div class="alert alert-success" style="background: var(--primary-soft); color: var(--primary); padding: 12px 16px; border-radius: 8px; margin-bottom: 16px; border: 1px solid var(--primary);">
    {{ session('success') }}
</div>
@endif

@if(session('error'))
<div class="alert alert-error" style="background: rgba(239, 68, 68, 0.15); color: #ef4444; padding: 12px 16px; border-radius: 8px; margin-bottom: 16px; border: 1px solid rgba(239, 68, 68, 0.3);">
    {{ session('error') }}
</div>
@endif

@if($errors->any())
<div class="alert alert-error" style="background: rgba(239, 68, 68, 0.15); color: #ef4444; padding: 12px 16px; border-radius: 8px; margin-bottom: 16px; border: 1px solid rgba(239, 68, 68, 0.3);">
    @foreach($errors->all() as $error)
        <div>{{ $error }}</div>
    @endforeach
</div>
@endif

<div class="panel-card">
    <div class="panel-header">
        <h2>{{ __('admin.deposits.title') }}</h2>
        <form method="GET" action="{{ route('admin.deposits') }}" class="search-form">
            <select name="status">
                <option value="">{{ __('admin.users.all_status') }}</option>
                <option value="pending" {{ request('status') === 'pending' ? 'selected' : '' }}>{{ __('admin.deposits.pending') }}</option>
                <option value="confirmed" {{ request('status') === 'confirmed' ? 'selected' : '' }}>{{ __('admin.deposits.confirmed') }}</option>
                <option value="rejected" {{ request('status') === 'rejected' ? 'selected' : '' }}>{{ __('admin.deposits.rejected') }}</option>
            </select>
            <button type="submit">{{ __('admin.deposits.filter') }}</button>
        </form>
    </div>

    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>{{ __('admin.table.id') }}</th>
                    <th>{{ __('admin.deposits.user') }}</th>
                    <th>{{ __('admin.deposits.network') }}</th>
                    <th>{{ __('admin.deposits.amount') }}</th>
                    <th>{{ __('admin.deposits.voucher') }}</th>
                    <th>{{ __('admin.deposits.status') }}</th>
                    <th>{{ __('admin.deposits.date') }}</th>
                    <th>{{ __('admin.table.actions') }}</th>
                </tr>
            </thead>
            <tbody>
                @forelse($deposits as $deposit)
                <tr>
                    <td>{{ str_pad($deposit->user_id ?? 0, 6, '0', STR_PAD_LEFT) }}</td>
                    <td>{{ substr($deposit->user->wallet_address ?? 'N/A', 0, 15) }}...</td>
                    <td>{{ $deposit->network }}</td>
                    <td>${{ number_format($deposit->amount, 2) }}</td>
                    <td>
                        @if($deposit->voucher_path)
                        <div style="display: flex; gap: 6px; align-items: center;">
                            <button type="button" class="btn btn-sm btn-info" onclick="viewVoucher('{{ asset('storage/' . $deposit->voucher_path) }}')">{{ __('admin.deposits.view') }}</button>
                            <a href="{{ asset('storage/' . $deposit->voucher_path) }}" download class="btn btn-sm btn-outline-info">{{ __('admin.deposits.download') }}</a>
                        </div>
                        @else
                        <span class="text-muted">{{ __('admin.deposits.no_voucher') }}</span>
                        @endif
                    </td>
                    <td>
                        <span class="badge badge-{{ $deposit->status === 'confirmed' ? 'success' : ($deposit->status === 'rejected' ? 'danger' : 'warning') }}">
                            {{ __('admin.deposits.' . $deposit->status) }}
                        </span>
                    </td>
                    <td>{{ $deposit->created_at->format('Y-m-d H:i') }}</td>
                    <td>
                        @if($deposit->status === 'pending')
                        <div style="display: flex; flex-direction: column; gap: 8px;">
                            <button type="button" class="btn btn-sm btn-success" onclick="showConfirmModal({{ $deposit->id }}, {{ $deposit->amount }}, '{{ $deposit->user->wallet_address ?? 'N/A' }}')">{{ __('admin.deposits.confirm') }}</button>
                            <form method="POST" action="{{ route('admin.deposits.update', $deposit->id) }}" style="display:inline-block;">
                                @csrf
                                @method('PUT')
                                <input type="hidden" name="status" value="rejected">
                                <textarea name="reason" placeholder="{{ __('admin.deposits.rejection_reason') }}" required class="form-control-sm" rows="2" style="width: 200px; margin-bottom: 6px;"></textarea>
                                <button type="submit" class="btn btn-sm btn-danger">{{ __('admin.deposits.reject') }}</button>
                            </form>
                        </div>
                        @else
                        <span class="text-muted">{{ __('admin.deposits.' . $deposit->status) }}</span>
                        @if($deposit->admin_note)
                        <br><small class="text-muted" style="font-size: 11px;">{{ Str::limit($deposit->admin_note, 30) }}</small>
                        @endif
                        @endif
                    </td>
                </tr>
                @empty
                <tr>
                    <td colspan="7" class="text-center">{{ __('admin.deposits.no_found') }}</td>
                </tr>
                @endforelse
            </tbody>
        </table>
    </div>

    @include('admin.partials.pagination', ['items' => $deposits])
</div>

<!-- Confirm Deposit Modal -->
<div id="confirmModal" class="modal-backdrop" style="display: none;" onclick="closeConfirmModal()">
    <div class="modal-card" onclick="event.stopPropagation();">
        <div class="modal-header">
            <h3 class="modal-title">{{ __('admin.deposits.confirm_modal_title') }}</h3>
        </div>
        <div class="modal-body">
            <p style="color: var(--text); margin: 0 0 12px 0;">{{ __('admin.deposits.confirm_question') }}</p>
            <div style="background: rgba(255, 255, 255, 0.05); padding: 12px; border-radius: 8px; margin-bottom: 12px;">
                <div style="display: flex; justify-content: space-between; margin-bottom: 6px;">
                    <span style="color: var(--muted); font-size: 13px;">{{ __('admin.deposits.amount') }}:</span>
                    <span style="color: var(--text); font-weight: 600;" id="confirmAmount">$0.00</span>
                </div>
                <div style="display: flex; justify-content: space-between;">
                    <span style="color: var(--muted); font-size: 13px;">{{ __('admin.deposits.user') }}:</span>
                    <span style="color: var(--text); font-size: 13px;" id="confirmUser">N/A</span>
                </div>
            </div>
            <p style="color: var(--muted); margin: 0; font-size: 12px;">{{ __('admin.deposits.confirm_note') }}</p>
        </div>
        <div class="modal-footer">
            <button type="button" class="btn btn-outline" onclick="closeConfirmModal()">{{ __('admin.common.cancel') }}</button>
            <form id="confirmForm" method="POST" style="display: inline;">
                @csrf
                @method('PUT')
                <input type="hidden" name="status" value="confirmed">
                <button type="submit" class="btn btn-success">{{ __('admin.deposits.confirm_btn') }}</button>
            </form>
        </div>
    </div>
</div>

<!-- Voucher View Modal -->
<div id="voucherModal" class="modal-backdrop" style="display: none;" onclick="closeVoucherModal()">
    <div class="modal-card voucher-modal" onclick="event.stopPropagation();">
        <div class="modal-header">
            <h3>{{ __('admin.deposits.transaction_voucher') }}</h3>
            <button type="button" class="modal-close" onclick="closeVoucherModal()">×</button>
        </div>
        <div class="modal-body voucher-modal-body">
            <img id="voucherImage" src="" alt="Voucher" class="voucher-image">
        </div>
    </div>
</div>

<script>
let currentConfirmDepositId = null;

function showConfirmModal(depositId, amount, userAddress) {
    currentConfirmDepositId = depositId;
    document.getElementById('confirmAmount').textContent = '$' + parseFloat(amount).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    document.getElementById('confirmUser').textContent = userAddress ? (userAddress.substring(0, 10) + '...') : 'N/A';
    document.getElementById('confirmForm').action = '{{ route("admin.deposits.update", ":id") }}'.replace(':id', depositId);
    document.getElementById('confirmModal').style.display = 'flex';
}

function closeConfirmModal() {
    document.getElementById('confirmModal').style.display = 'none';
    currentConfirmDepositId = null;
}

function viewVoucher(imageUrl) {
    document.getElementById('voucherImage').src = imageUrl;
    document.getElementById('voucherModal').style.display = 'flex';
}

function closeVoucherModal() {
    document.getElementById('voucherModal').style.display = 'none';
}

// Close confirm modal on backdrop click
document.getElementById('confirmModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeConfirmModal();
    }
});
</script>
@endsection

