<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        if (DB::getDriverName() === 'sqlite') {
            // SQLite: Drop and recreate CHECK constraint for status column
            // Since SQLite doesn't support modifying CHECK constraints directly,
            // we need to recreate the table or use a workaround
            
            // Get current table structure
            $pdo = DB::connection()->getPdo();
            
            // Check if constraint exists by trying to insert 'closed' value
            // If it fails, we need to fix the constraint
            try {
                // Create a temporary table with updated constraint
                DB::statement("
                    CREATE TABLE ai_trade_bots_new (
                        id INTEGER PRIMARY KEY AUTOINCREMENT,
                        user_id INTEGER NOT NULL,
                        name VARCHAR(255) NOT NULL,
                        trading_pair VARCHAR(255) DEFAULT 'BTC/USDT',
                        leverage DECIMAL(5,2) DEFAULT 1,
                        amount DECIMAL(20,8) NOT NULL,
                        weight_range_min DECIMAL(5,2) DEFAULT 0,
                        weight_range_max DECIMAL(5,2) DEFAULT 10,
                        quota_range_min DECIMAL(20,8) DEFAULT 500,
                        quota_range_max DECIMAL(20,8) DEFAULT 10000,
                        max_profit_percent DECIMAL(5,2) DEFAULT 3.00,
                        max_loss_percent DECIMAL(5,2) DEFAULT 1.50,
                        expected_return_min DECIMAL(5,2) DEFAULT 1.50,
                        expected_return_max DECIMAL(5,2) DEFAULT 3.00,
                        status VARCHAR(255) DEFAULT 'active' CHECK(status IN ('active', 'paused', 'stopped', 'closed')),
                        strategy_config TEXT,
                        total_profit DECIMAL(20,8) DEFAULT 0,
                        total_loss DECIMAL(20,8) DEFAULT 0,
                        created_at TIMESTAMP,
                        updated_at TIMESTAMP,
                        closed_at TIMESTAMP,
                        close_reason TEXT,
                        admin_set_trading_time_minutes INTEGER,
                        admin_activated_at TIMESTAMP,
                        time_based_profit_loss_rules TEXT
                    )
                ");
                
                // Copy data from old table to new table
                DB::statement("
                    INSERT INTO ai_trade_bots_new 
                    SELECT * FROM ai_trade_bots
                ");
                
                // Drop old table
                DB::statement("DROP TABLE ai_trade_bots");
                
                // Rename new table
                DB::statement("ALTER TABLE ai_trade_bots_new RENAME TO ai_trade_bots");
                
                // Recreate indexes
                DB::statement("CREATE INDEX ai_trade_bots_user_id_status_index ON ai_trade_bots(user_id, status)");
                
            } catch (\Exception $e) {
                // If migration fails, log and continue
                // The application can still work if we handle this at the code level
                \Illuminate\Support\Facades\Log::warning("SQLite status constraint migration failed: " . $e->getMessage());
            }
        } else {
            // For MySQL/MariaDB, enum is already handled by previous migration
            // Just ensure the enum includes 'closed'
            try {
                DB::statement("ALTER TABLE ai_trade_bots MODIFY COLUMN status ENUM('active', 'paused', 'stopped', 'closed') DEFAULT 'active'");
            } catch (\Exception $e) {
                // If it fails, the enum might already be correct
                \Illuminate\Support\Facades\Log::info("Status enum update skipped (might already be correct): " . $e->getMessage());
            }
        }
    }

    public function down(): void
    {
        // Revert: Remove 'closed' from enum
        if (DB::getDriverName() === 'sqlite') {
            // For SQLite, would need to recreate table again
            // This is complex, so we'll skip down migration for SQLite
            return;
        }
        
        DB::statement("ALTER TABLE ai_trade_bots MODIFY COLUMN status ENUM('active', 'paused', 'stopped') DEFAULT 'active'");
    }
};

