<?php

namespace App\Services;

use App\Models\AiTradeBot;
use App\Models\Order;
use App\Models\User;
use App\Services\BinanceService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OrderService
{
    protected $binanceService;

    public function __construct(BinanceService $binanceService)
    {
        $this->binanceService = $binanceService;
    }

    /**
     * Process active bots and create orders
     */
    public function processActiveBots(): array
    {
        $activeBots = AiTradeBot::where('status', 'active')
            ->with('user')
            ->get();

        $results = [
            'processed' => 0,
            'orders_created' => 0,
            'bots_stopped' => 0,
            'errors' => 0,
        ];

        foreach ($activeBots as $bot) {
            try {
                $result = $this->processBot($bot);
                $results['processed']++;
                $results['orders_created'] += $result['orders_created'] ?? 0;
                $results['bots_stopped'] += $result['bot_stopped'] ? 1 : 0;
            } catch (\Exception $e) {
                Log::error("Error processing bot #{$bot->id}: " . $e->getMessage());
                $results['errors']++;
            }
        }

        return $results;
    }

    /**
     * Process a single bot
     */
    public function processBot(AiTradeBot $bot, bool $force = false): array
    {
        $result = [
            'orders_created' => 0,
            'bot_stopped' => false,
        ];

        // Get current market price (needed for order creation and for final-minute order before close)
        $currentPrice = $this->binanceService->getCurrentPrice($bot->trading_pair);
        if (!$currentPrice) {
            Log::warning("Could not get price for {$bot->trading_pair}");
            return $result;
        }

        // Check if trading time has completed: catch up any missed-minute orders, then final order, then auto-close
        if ($bot->admin_activated_at) {
            // Elapsed minutes since activation (always positive; use absolute diff so timezone-safe)
            $elapsedMinutes = (int) $bot->admin_activated_at->diffInMinutes(now(), true);
            $totalTime = (int) ($bot->admin_set_trading_time_minutes ?? 1440); // Default 24 hours
            
            Log::info("Checking bot expiration", [
                'bot_id' => $bot->id,
                'admin_activated_at' => $bot->admin_activated_at?->toDateTimeString(),
                'now' => now()->toDateTimeString(),
                'admin_set_trading_time_minutes' => $totalTime,
                'elapsed_minutes' => $elapsedMinutes,
                'status' => $bot->status,
                'should_close' => $elapsedMinutes >= $totalTime && $bot->status === 'active',
            ]);
            
            if ($elapsedMinutes >= $totalTime && $bot->status === 'active') {
                Log::info("Bot expired, auto-completing", [
                    'bot_id' => $bot->id,
                    'elapsed_minutes' => $elapsedMinutes,
                    'total_time' => $totalTime,
                ]);
                
                $lastOrder = $bot->orders()->where('status', 'filled')->latest('executed_at')->first();
                $lastMinute = $lastOrder && $bot->admin_activated_at
                    ? (int) abs($bot->admin_activated_at->diffInMinutes($lastOrder->executed_at))
                    : -1;
                // Create orders for every missed minute (so P/L is correct even if cron didn't run every minute)
                for ($m = $lastMinute + 1; $m <= $totalTime; $m++) {
                    $order = $this->createOrder($bot->fresh(), $currentPrice);
                    if ($order) {
                        $this->executeOrder($order, $currentPrice, $m);
                        $result['orders_created']++;
                    }
                }
                $this->autoCompleteBot($bot->fresh());
                $result['bot_stopped'] = true;
                return $result;
            }
        }

        // Check if bot should create an order based on trading interval
        // If force=true (admin update/manual), bypass interval check and process immediately
        if (!$force && !$this->shouldCreateOrder($bot)) {
            return $result;
        }

        // Check profit/loss limits
        $netProfitLoss = $bot->total_profit - $bot->total_loss;
        $profitPercent = ($netProfitLoss / $bot->amount) * 100;
        $lossPercent = abs(($netProfitLoss / $bot->amount) * 100);

        // Auto-stop if max profit reached
        if ($profitPercent >= $bot->max_profit_percent) {
            $this->autoStopBot($bot, 'Max profit reached: ' . number_format($profitPercent, 2) . '%');
            $result['bot_stopped'] = true;
            return $result;
        }

        // Auto-stop if max loss reached
        if ($netProfitLoss < 0 && $lossPercent >= $bot->max_loss_percent) {
            $this->autoStopBot($bot, 'Max loss reached: ' . number_format($lossPercent, 2) . '%');
            $result['bot_stopped'] = true;
            return $result;
        }

        // Create order
        $order = $this->createOrder($bot, $currentPrice);
        if ($order) {
            $result['orders_created'] = 1;
            
            // Execute order immediately (simulate market order)
            $this->executeOrder($order, $currentPrice);
        }

        return $result;
    }

    /**
     * Check if order should be created based on trading interval.
     * When admin has set time-based rules (first period / remaining period), use 1-minute interval
     * so P/L accumulates correctly over the period. Otherwise use system default_trading_interval.
     */
    protected function shouldCreateOrder(AiTradeBot $bot): bool
    {
        $lastOrder = $bot->orders()
            ->where('status', 'filled')
            ->latest('executed_at')
            ->first();

        if (!$lastOrder) {
            return true;
        }

        // Admin activated bot: run orders every 30 seconds so P/L updates more often (gradual display)
        if ($bot->admin_activated_at && $bot->admin_set_trading_time_minutes) {
            $intervalSeconds = 30;
        } else {
            $systemSetting = \App\Models\SystemSetting::where('key', 'default_trading_interval')->first();
            $interval = $systemSetting ? $systemSetting->value : '15m';
            $intervalSeconds = $this->intervalToSeconds($interval);
        }

        $timeSinceLastOrder = now()->diffInSeconds($lastOrder->executed_at);
        return $timeSinceLastOrder >= $intervalSeconds;
    }

    /**
     * Convert interval string to seconds
     */
    protected function intervalToSeconds(string $interval): int
    {
        $intervals = [
            '1m' => 60,
            '5m' => 300,
            '15m' => 900,
            '1h' => 3600,
            '4h' => 14400,
            '1d' => 86400,
        ];

        return $intervals[$interval] ?? 900; // Default to 15 minutes
    }

    /**
     * Create an order for a bot
     */
    protected function createOrder(AiTradeBot $bot, float $currentPrice): ?Order
    {
        try {
            // Determine order side (buy/sell) based on strategy
            // Simple strategy: alternate between buy and sell, or use market trend
            $lastOrder = $bot->orders()->latest('created_at')->first();
            $side = $lastOrder && $lastOrder->side === 'buy' ? 'sell' : 'buy';

            // Calculate order amount (use a portion of bot amount)
            $orderAmount = $bot->amount * 0.1; // 10% of bot amount per order

            $order = Order::create([
                'user_id' => $bot->user_id,
                'ai_trade_bot_id' => $bot->id,
                'order_type' => 'market',
                'side' => $side,
                'trading_pair' => $bot->trading_pair,
                'amount' => $orderAmount,
                'price' => $currentPrice,
                'status' => 'pending',
            ]);

            return $order;
        } catch (\Exception $e) {
            Log::error("Error creating order for bot #{$bot->id}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Execute an order (simulate market order execution).
     * @param int|null $elapsedMinutesOverride When set, used for time-based P/L as if the order ran at that minute (for catch-up).
     */
    protected function executeOrder(Order $order, float $currentPrice, ?int $elapsedMinutesOverride = null): void
    {
        try {
            DB::transaction(function () use ($order, $currentPrice, $elapsedMinutesOverride) {
                // Update order status to processing
                $order->update([
                    'status' => 'processing',
                ]);

                // Simulate order execution with slight price variation
                $executionPrice = $this->calculateExecutionPrice($currentPrice, $order->side);
                $filledAmount = $order->amount;

                // Calculate profit/loss (optionally for a specific minute when catching up)
                $profitLoss = $this->calculateProfitLoss($order, $executionPrice, $elapsedMinutesOverride);

                // Update order
                $order->update([
                    'status' => 'filled',
                    'filled_amount' => $filledAmount,
                    'price' => $executionPrice,
                    'profit_loss' => $profitLoss,
                    'executed_at' => now(),
                ]);

                // Update bot profit/loss
                $bot = $order->aiTradeBot;
                if ($bot) {
                    if ($profitLoss > 0) {
                        $bot->increment('total_profit', $profitLoss);
                    } else {
                        $bot->increment('total_loss', abs($profitLoss));
                    }
                }

                // Update user's USDT balance with profit/loss
                $user = $order->user;
                $usdtBalance = \App\Models\UserBalance::where('user_id', $user->id)
                    ->where('coin_name', 'USDT')
                    ->first();
                
                if ($usdtBalance) {
                    // Credit or debit USDT balance based on profit/loss
                    if ($profitLoss > 0) {
                        $usdtBalance->increment('balance', $profitLoss);
                        // Also update legacy balance for backward compatibility
                        $user->increment('balance', $profitLoss);
                    } elseif ($profitLoss < 0) {
                        // Deduct loss from balance (don't go below 0)
                        $newBalance = max(0, $usdtBalance->balance + $profitLoss); // profitLoss is negative, so add it
                        $usdtBalance->update(['balance' => $newBalance]);
                        // Also update legacy balance for backward compatibility
                        $newLegacyBalance = max(0, $user->balance + $profitLoss);
                        $user->update(['balance' => $newLegacyBalance]);
                    }
                } else {
                    // If UserBalance doesn't exist, create it and credit profit (if positive)
                    if ($profitLoss > 0) {
                        \App\Models\UserBalance::create([
                            'user_id' => $user->id,
                            'coin_name' => 'USDT',
                            'balance' => $profitLoss,
                            'locked_balance' => 0,
                        ]);
                        // Also update legacy balance for backward compatibility
                        $user->increment('balance', $profitLoss);
                    }
                }

                // Update daily earnings
                $this->updateDailyEarnings($order->user_id, $profitLoss);
            });
        } catch (\Exception $e) {
            Log::error("Error executing order #{$order->id}: " . $e->getMessage());
            $order->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Calculate execution price with slight variation
     */
    protected function calculateExecutionPrice(float $currentPrice, string $side): float
    {
        // Add small random variation (±0.1%)
        $variation = (rand(-10, 10) / 10000); // ±0.1%
        
        if ($side === 'buy') {
            // Buy orders execute slightly above market (slippage)
            return $currentPrice * (1 + abs($variation));
        } else {
            // Sell orders execute slightly below market (slippage)
            return $currentPrice * (1 - abs($variation));
        }
    }

    /**
     * Calculate profit/loss for an order.
     * Uses time-based rules if admin has set them, otherwise uses market-based calculation.
     * @param int|null $elapsedMinutesOverride When set, used as elapsed minute for time-based P/L (catch-up).
     */
    protected function calculateProfitLoss(Order $order, float $executionPrice, ?int $elapsedMinutesOverride = null): float
    {
        $bot = $order->aiTradeBot->fresh(); // Refresh to get latest bot data
        
        // If bot is activated by admin, use gradual profit/loss calculation (default 24hr if trade time not set)
        if ($bot && $bot->admin_activated_at) {
            Log::info("Using gradual P/L calculation", [
                'order_id' => $order->id,
                'bot_id' => $bot->id,
                'elapsed_override' => $elapsedMinutesOverride,
            ]);
            return $this->calculateGradualProfitLoss($bot, $order, $elapsedMinutesOverride);
        }
        
        Log::info("Using market-based calculation", [
            'order_id' => $order->id,
            'bot_id' => $bot?->id,
            'has_activated_at' => $bot ? !is_null($bot->admin_activated_at) : false,
            'has_trading_time' => $bot ? !is_null($bot->admin_set_trading_time_minutes) : false,
        ]);

        // Default: Calculate P/L based on market prices
        // Get previous order for this bot to calculate P/L
        $previousOrder = Order::where('ai_trade_bot_id', $order->ai_trade_bot_id)
            ->where('id', '<', $order->id)
            ->where('status', 'filled')
            ->latest('executed_at')
            ->first();

        if (!$previousOrder) {
            // First order, no P/L yet
            return 0;
        }

        // Calculate P/L based on order side
        if ($order->side === 'sell' && $previousOrder->side === 'buy') {
            // Selling after buying: profit = (sell_price - buy_price) * amount
            $profitLoss = ($executionPrice - $previousOrder->price) * $order->amount;
        } elseif ($order->side === 'buy' && $previousOrder->side === 'sell') {
            // Buying after selling: profit = (sell_price - buy_price) * amount
            $profitLoss = ($previousOrder->price - $executionPrice) * $order->amount;
        } else {
            // Same side orders, no P/L
            return 0;
        }

        return round($profitLoss, 8);
    }

    /**
     * Calculate gradual profit/loss for order execution.
     * Distributes admin set win/loss % over trading time with random variation.
     * @param int|null $elapsedMinutesOverride When set (e.g. catch-up), use this as elapsed minute instead of now().
     */
    protected function calculateGradualProfitLoss(AiTradeBot $bot, Order $order, ?int $elapsedMinutesOverride = null): float
    {
        if (!$bot->admin_activated_at) {
            return 0;
        }

        // Elapsed minutes: override for catch-up, otherwise since activation (absolute for consistency)
        $elapsedMinutes = $elapsedMinutesOverride !== null
            ? $elapsedMinutesOverride
            : (int) $bot->admin_activated_at->diffInMinutes(now(), true);
        
        $totalTime = (int) ($bot->admin_set_trading_time_minutes ?? 1440); // Default 24 hours
        $amount = (float) $bot->amount;
        $maxProfitPercent = (float) ($bot->max_profit_percent ?? 0);
        $maxLossPercent = (float) ($bot->max_loss_percent ?? 0);
        
        if ($totalTime <= 0 || $amount <= 0) {
            return 0;
        }
        
        // Calculate progress (0 to 1)
        $progress = min(1, max(0, $elapsedMinutes / $totalTime));
        
        // Target final values based on admin set percentages
        $targetProfit = ($maxProfitPercent / 100) * $amount;
        $targetLoss = ($maxLossPercent / 100) * $amount;
        
        // Calculate incremental P/L per order (distribute over estimated orders)
        // Estimate ~1 order per minute
        $estimatedOrders = max(1, $totalTime);
        
        // Add random variation (±10% variation for realistic fluctuation)
        $variation = 0.9 + (rand(0, 2000) / 10000); // 0.9 to 1.1 (±10%)
        
        // Calculate incremental profit/loss for this order
        // Distribute target profit/loss over orders with random variation
        $orderProfit = ($targetProfit / $estimatedOrders) * $variation;
        $orderLoss = ($targetLoss / $estimatedOrders) * $variation;
        
        // Randomly decide if this order is profit or loss (based on target ratio)
        $profitRatio = $targetProfit / ($targetProfit + $targetLoss);
        if ($targetProfit + $targetLoss == 0) {
            $profitRatio = 0.5; // Default 50/50 if both are 0
        }
        
        // Use random to decide profit or loss for this order
        $isProfit = (rand(0, 10000) / 10000) < $profitRatio;
        
        if ($isProfit && $orderProfit > 0) {
            return round($orderProfit, 8);
        } elseif (!$isProfit && $orderLoss > 0) {
            return round(-$orderLoss, 8); // Negative for loss
        }
        
        return 0;
    }

    /**
     * Update daily earnings for user
     */
    protected function updateDailyEarnings(int $userId, float $profitLoss): void
    {
        try {
            $today = now()->startOfDay();
            
            $earning = \App\Models\DailyEarning::firstOrCreate(
                ['user_id' => $userId, 'date' => $today],
                [
                    'total_earnings' => 0,
                    'total_profit' => 0,
                    'total_loss' => 0,
                    'total_trades' => 0,
                    'winning_trades' => 0,
                    'losing_trades' => 0,
                ]
            );

            if ($profitLoss > 0) {
                $earning->increment('total_profit', $profitLoss);
                $earning->increment('winning_trades');
            } else {
                $earning->increment('total_loss', abs($profitLoss));
                $earning->increment('losing_trades');
            }

            $earning->increment('total_trades');
            $earning->update([
                'total_earnings' => $earning->total_profit - $earning->total_loss,
            ]);
        } catch (\Exception $e) {
            Log::error("Error updating daily earnings for user #{$userId}: " . $e->getMessage());
        }
    }

    /**
     * Auto-complete bot when trading time finishes
     */
    protected function autoCompleteBot(AiTradeBot $bot): void
    {
        try {
            DB::transaction(function () use ($bot) {
                $user = $bot->user->fresh();
                
                // Calculate final P/L based on admin set win/loss %
                // Real trading logic: Final Net P/L = (Win % - Loss %) * amount
                $maxProfitPercent = (float) ($bot->max_profit_percent ?? 0);
                $maxLossPercent = (float) ($bot->max_loss_percent ?? 0);
                
                // Final Net P/L = (Win % - Loss %) * amount
                $netPercent = $maxProfitPercent - $maxLossPercent;
                $finalNetPL = ($netPercent / 100) * $bot->amount;
                
                // Real trading logic: Only show profit OR loss, not both
                $finalProfit = 0;
                $finalLoss = 0;
                
                if ($finalNetPL > 0) {
                    // Net P/L is positive → only profit (no loss)
                    $finalProfit = $finalNetPL;
                    $finalLoss = 0;
                } elseif ($finalNetPL < 0) {
                    // Net P/L is negative → only loss (no profit)
                    $finalProfit = 0;
                    $finalLoss = abs($finalNetPL);
                }
                
                // Calculate adjustment BEFORE updating bot (to get current P/L)
                $currentNetPL = (float) $bot->total_profit - (float) $bot->total_loss;
                $adjustmentNeeded = $finalNetPL - $currentNetPL;
                
                Log::info("Bot auto-complete: Balance adjustment calculation", [
                    'bot_id' => $bot->id,
                    'current_profit' => $bot->total_profit,
                    'current_loss' => $bot->total_loss,
                    'current_net_pl' => $currentNetPL,
                    'final_net_pl' => $finalNetPL,
                    'adjustment_needed' => $adjustmentNeeded,
                ]);
                
                // Set final profit/loss directly
                $bot->update([
                    'total_profit' => $finalProfit,
                    'total_loss' => $finalLoss,
                ]);
                
                // Get USDT balance once
                $usdtBalance = \App\Models\UserBalance::where('user_id', $user->id)
                    ->where('coin_name', 'USDT')
                    ->first();
                
                // Update balance with final adjustment (ALWAYS update, even if 0, to ensure balance reflects final P/L)
                if ($usdtBalance) {
                    if ($adjustmentNeeded > 0) {
                        $usdtBalance->increment('balance', $adjustmentNeeded);
                        // Also update legacy balance for backward compatibility
                        $user->increment('balance', $adjustmentNeeded);
                        Log::info("Bot auto-complete: Added profit to balance", [
                            'bot_id' => $bot->id,
                            'adjustment' => $adjustmentNeeded,
                            'new_balance' => $usdtBalance->fresh()->balance,
                        ]);
                    } elseif ($adjustmentNeeded < 0) {
                        // Deduct loss from balance (don't go below 0)
                        $newBalance = max(0, $usdtBalance->balance + $adjustmentNeeded);
                        $usdtBalance->update(['balance' => $newBalance]);
                        // Also update legacy balance for backward compatibility
                        $newLegacyBalance = max(0, $user->balance + $adjustmentNeeded);
                        $user->update(['balance' => $newLegacyBalance]);
                        Log::info("Bot auto-complete: Deducted loss from balance", [
                            'bot_id' => $bot->id,
                            'adjustment' => $adjustmentNeeded,
                            'new_balance' => $usdtBalance->fresh()->balance,
                        ]);
                    } else {
                        Log::info("Bot auto-complete: No balance adjustment needed", [
                            'bot_id' => $bot->id,
                            'current_net_pl' => $currentNetPL,
                            'final_net_pl' => $finalNetPL,
                        ]);
                    }
                }
                
                // Update daily earnings with final adjustment (if any)
                if ($adjustmentNeeded != 0) {
                    $this->updateDailyEarnings($user->id, $adjustmentNeeded);
                }
                
                // Unlock UserBalance locked_balance
                if ($usdtBalance) {
                    $currentLocked = max(0, (float) $usdtBalance->locked_balance);
                    $unlockAmount = min($currentLocked, $bot->amount);
                    if ($unlockAmount > 0) {
                        $usdtBalance->decrement('locked_balance', $unlockAmount);
                    }
                }
                
                // Also update legacy balance for backward compatibility
                $currentLocked = max(0, (float) $user->locked_balance);
                $unlockAmount = min($currentLocked, $bot->amount);
                if ($unlockAmount > 0) {
                    $user->decrement('locked_balance', $unlockAmount);
                }

                // Normalize order-level profit_loss so Order Record sum matches Total Profit/Loss (admin set %)
                $this->normalizeOrdersToMatchFinalPL($bot, $finalNetPL);

                // Close bot
                $bot->update([
                    'status' => 'closed',
                    'closed_at' => now(),
                    'close_reason' => 'Trading time completed (' . $bot->admin_set_trading_time_minutes . ' minutes). Final result: ' . number_format($netPercent, 2) . '%',
                ]);

                if ($finalNetPL > 0) {
                    \App\Services\NotificationService::createBotProfit($user, $bot->name ?? ('Bot #' . $bot->id), $finalNetPL);
                }
                
                Log::info("Bot auto-completed", [
                    'bot_id' => $bot->id,
                    'final_net_pl' => $finalNetPL,
                    'adjustment' => $adjustmentNeeded,
                    'net_percent' => $netPercent,
                ]);
            });
        } catch (\Exception $e) {
            Log::error("Error auto-completing bot #{$bot->id}: " . $e->getMessage());
        }
    }

    /**
     * Normalize each order's profit_loss so that sum(orders.profit_loss) = finalNetPL.
     * Preserves buy/sell and win/loss pattern (scale factor). Order Record will match bot summary.
     */
    public function normalizeOrdersToMatchFinalPL(AiTradeBot $bot, float $finalNetPL): void
    {
        $orders = $bot->orders()->where('status', 'filled')->orderBy('executed_at')->get();
        if ($orders->isEmpty()) {
            return;
        }

        $currentSum = $orders->sum(fn ($o) => (float) $o->profit_loss);
        $n = $orders->count();
        $epsilon = 1e-9;

        if (abs($currentSum) < $epsilon) {
            // Distribute finalNetPL equally so order history total matches summary
            $perOrder = round($finalNetPL / $n, 8);
            foreach ($orders as $order) {
                $order->update(['profit_loss' => $perOrder]);
            }
            return;
        }

        $scale = $finalNetPL / $currentSum;
        $newValues = [];
        foreach ($orders as $order) {
            $newValues[] = round((float) $order->profit_loss * $scale, 8);
        }
        // Fix rounding: last order gets adjusted so sum is exactly finalNetPL
        $sumSoFar = array_sum(array_slice($newValues, 0, -1));
        $newValues[$n - 1] = round($finalNetPL - $sumSoFar, 8);

        foreach ($orders as $i => $order) {
            $order->update(['profit_loss' => $newValues[$i]]);
        }

        Log::info('Orders normalized to match final P/L', [
            'bot_id' => $bot->id,
            'final_net_pl' => $finalNetPL,
            'orders_count' => $n,
        ]);
    }

    /**
     * Calculate final P/L for closed bot from admin-set win/loss % only.
     * Final Net P/L = (Win % - Loss %) * amount. Positive = profit only, negative = loss only.
     */
    public function calculateFinalPLFromWinLossPercent(AiTradeBot $bot): ?array
    {
        $amount = (float) $bot->amount;
        $maxProfitPercent = (float) ($bot->max_profit_percent ?? 0);
        $maxLossPercent = (float) ($bot->max_loss_percent ?? 0);

        if ($amount <= 0) {
            return null;
        }

        $netPercent = $maxProfitPercent - $maxLossPercent;
        $finalNetPL = ($netPercent / 100) * $amount;

        $totalProfit = $finalNetPL > 0 ? $finalNetPL : 0;
        $totalLoss = $finalNetPL < 0 ? abs($finalNetPL) : 0;

        return [
            'totalProfit' => $totalProfit,
            'totalLoss' => $totalLoss,
        ];
    }

    /**
     * Create one summary order so Order Record shows final P/L when bot is closed with no orders.
     */
    public function createSummaryOrdersForClosedBot(AiTradeBot $bot, float $finalNetPL): void
    {
        $orders = $bot->orders()->where('status', 'filled')->count();
        if ($orders > 0) {
            return;
        }

        $user = $bot->user;
        $amount = (float) $bot->amount;
        $tradingPair = $bot->trading_pair ?? 'BTCUSDT';
        $executedAt = now();

        Order::create([
            'user_id' => $user->id,
            'ai_trade_bot_id' => $bot->id,
            'order_type' => 'market',
            'side' => $finalNetPL >= 0 ? 'sell' : 'buy',
            'trading_pair' => $tradingPair,
            'amount' => $amount > 0 ? $amount : 1,
            'price' => 0,
            'filled_amount' => $amount > 0 ? $amount : 1,
            'status' => 'filled',
            'profit_loss' => round($finalNetPL, 8),
            'executed_at' => $executedAt,
        ]);

        Log::info('Summary order created for closed bot (no orders)', [
            'bot_id' => $bot->id,
            'final_net_pl' => $finalNetPL,
        ]);
    }

    /**
     * Auto-stop bot when limits reached
     */
    protected function autoStopBot(AiTradeBot $bot, string $reason): void
    {
        try {
            DB::transaction(function () use ($bot, $reason) {
                $user = $bot->user;
                
                // Get USDT balance
                $usdtBalance = \App\Models\UserBalance::where('user_id', $user->id)
                    ->where('coin_name', 'USDT')
                    ->first();
                
                // Unlock UserBalance locked_balance
                if ($usdtBalance) {
                    $currentLocked = max(0, (float) $usdtBalance->locked_balance);
                    $unlockAmount = min($currentLocked, $bot->amount);
                    if ($unlockAmount > 0) {
                        $usdtBalance->decrement('locked_balance', $unlockAmount);
                    }
                }
                
                // Also update legacy balance for backward compatibility
                // Use max() to prevent negative locked_balance
                $currentLocked = max(0, (float) $user->locked_balance);
                $unlockAmount = min($currentLocked, $bot->amount);
                if ($unlockAmount > 0) {
                    $user->decrement('locked_balance', $unlockAmount);
                }

                // Update bot status
                $bot->update([
                    'status' => 'stopped',
                    'close_reason' => $reason,
                ]);
            });
        } catch (\Exception $e) {
            Log::error("Error auto-stopping bot #{$bot->id}: " . $e->getMessage());
        }
    }
}

