<?php

namespace App\Services;

use App\Models\Notification;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class NotificationService
{
    public static function getTemplate(string $key): string
    {
        $row = DB::table('system_settings')->where('key', $key)->first();
        return $row ? (string) $row->value : self::defaultTemplate($key);
    }

    public static function defaultTemplate(string $key): string
    {
        $defaults = [
            'notification_welcome_title' => 'Welcome to AI Quantification Trading',
            'notification_welcome_message' => 'This AI trading bot is a future opportunity for every user. Earn up to 100% or 1–8.5% profits with risk reduced to 0%. Explore multiple features and grow with us.',
            'notification_deposit_approved_title' => 'Deposit Approved',
            'notification_deposit_approved_message' => 'Your deposit has been approved. Use this AI trading bot to improve your financial income and grow your assets with smart automation.',
            'notification_withdraw_approved_title' => 'Withdrawal Completed',
            'notification_withdraw_approved_message' => 'Your withdrawal has been completed successfully. Funds have been sent to your requested address.',
        ];
        return $defaults[$key] ?? '';
    }

    public static function createWelcome(User $user): void
    {
        $title = self::getTemplate('notification_welcome_title');
        $message = self::getTemplate('notification_welcome_message');
        Notification::create([
            'user_id' => $user->id,
            'type' => 'welcome',
            'title' => $title ?: 'Welcome to AI Quantification Trading',
            'message' => $message ?: 'This AI trading bot is a future opportunity for every user. Earn up to 100% or 1–8.5% profits with risk reduced to 0%. Explore multiple features and grow with us.',
        ]);
    }

    public static function createDepositApproved(User $user, float $amount, ?string $network = null): void
    {
        $title = self::getTemplate('notification_deposit_approved_title');
        $message = self::getTemplate('notification_deposit_approved_message');
        Notification::create([
            'user_id' => $user->id,
            'type' => 'deposit_approved',
            'title' => $title ?: 'Deposit Approved',
            'message' => $message ?: 'Your deposit has been approved. Use this AI trading bot to improve your financial income.',
            'data' => ['amount' => $amount, 'network' => $network],
        ]);
    }

    public static function createWithdrawApproved(User $user, float $amount, ?string $coin = null): void
    {
        $title = self::getTemplate('notification_withdraw_approved_title');
        $message = self::getTemplate('notification_withdraw_approved_message');
        Notification::create([
            'user_id' => $user->id,
            'type' => 'withdraw_approved',
            'title' => $title ?: 'Withdrawal Completed',
            'message' => $message ?: 'Your withdrawal has been completed successfully.',
            'data' => ['amount' => $amount, 'coin' => $coin],
        ]);
    }

    public static function createBotProfit(User $user, string $botName, float $profit): void
    {
        Notification::create([
            'user_id' => $user->id,
            'type' => 'bot_profit',
            'title' => 'Bot closed – profit credited',
            'message' => sprintf('Your bot "%s" has been closed. A profit of %s USDT has been credited to your balance.', $botName, number_format($profit, 2)),
            'data' => ['bot_name' => $botName, 'profit' => $profit],
        ]);
    }

    public static function createAdminMessage(User $user, string $type, string $title, string $message): void
    {
        Notification::create([
            'user_id' => $user->id,
            'type' => 'admin_' . $type,
            'title' => $title,
            'message' => $message,
            'data' => ['admin_type' => $type],
        ]);
    }
}
