<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Withdrawal;
use App\Models\UserBalance;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class WithdrawalController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();
        $withdrawals = $user->withdrawals()->latest()->paginate(20);

        return response()->json([
            'success' => true,
            'withdrawals' => $withdrawals
        ]);
    }

    public function store(Request $request)
    {
        $rules = [
            'coin_name' => 'required|string',
            'network' => 'required|string',
            'address' => 'required|string|min:10',
            'amount' => 'required|numeric|min:0.00000001',
        ];
        $user = $request->user();
        if (!empty($user->withdrawal_pin_hash)) {
            $rules['pin'] = 'required|string|digits:6';
        }
        $validator = Validator::make($request->all(), $rules, [
            'pin.required' => 'Security PIN is required to confirm withdrawal.',
            'pin.digits' => 'PIN must be 6 digits.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        if (!empty($user->withdrawal_pin_hash)) {
            if (!Hash::check($request->pin, $user->withdrawal_pin_hash)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid Security PIN. Withdrawal not confirmed.'
                ], 422);
            }
        }
        $coinName = strtoupper($request->coin_name);

        // Get user balance for the specific coin
        $userBalance = UserBalance::where('user_id', $user->id)
            ->where('coin_name', $coinName)
            ->first();

        if (!$userBalance) {
            return response()->json([
                'success' => false,
                'message' => 'No balance found for ' . $coinName
            ], 400);
        }

        // Check available balance (balance - locked_balance)
        $availableBalance = (float) $userBalance->balance - (float) $userBalance->locked_balance;
        if ($request->amount > $availableBalance) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance. Available: ' . number_format($availableBalance, 8) . ' ' . $coinName
            ], 400);
        }

        // Create withdrawal
        $withdrawal = DB::transaction(function () use ($user, $userBalance, $request, $coinName) {
            // Lock balance in UserBalance
            $userBalance->increment('locked_balance', $request->amount);
            
            // Also update legacy locked_balance in users table for backward compatibility
            $user->increment('locked_balance', $request->amount);

            return Withdrawal::create([
                'user_id' => $user->id,
                'coin_name' => $coinName,
                'network' => $request->network,
                'address' => $request->address,
                'amount' => $request->amount,
                'fee' => 0, // Calculate fee if needed
                'status' => 'pending',
            ]);
        });

        return response()->json([
            'success' => true,
            'withdrawal' => $withdrawal,
            'message' => 'Withdrawal request submitted successfully'
        ], 201);
    }
}
