<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Services\WalletService;
use Illuminate\Support\Facades\Validator;

class WalletController extends Controller
{
    protected $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
    }

    public function connect(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'wallet_address' => 'required|string',
            'wallet_type' => 'required|string|in:metamask,coinbase,trust,okx,imtoken,onchain,defi',
            'invite_code' => 'nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $result = $this->walletService->connect(
            $request->wallet_address,
            $request->wallet_type,
            $request->invite_code
        );

        if ($result['success']) {
            // Create session (for API stateless, we don't use session)
            // Return user data for frontend to store in localStorage/sessionStorage

            return response()->json([
                'success' => true,
                'user' => [
                    'id' => $result['user']->id,
                    'wallet_address' => $result['user']->wallet_address,
                    'balance' => $result['user']->balance,
                    'locked_balance' => $result['user']->locked_balance,
                    'account_weight_score' => $result['user']->account_weight_score,
                    'authorization_quota' => $result['user']->authorization_quota,
                    'available_quota' => $result['user']->available_quota,
                    'invite_code' => $result['user']->invite_code,
                ],
                'message' => $result['message']
            ]);
        }

        return response()->json($result, 400);
    }

    public function disconnect(Request $request)
    {
        session()->forget(['wallet_address', 'user_id', 'wallet.connected']);
        
        return response()->json([
            'success' => true,
            'message' => 'Wallet disconnected'
        ]);
    }
}
