<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Referral;

class UserController extends Controller
{
    public function profile(Request $request)
    {
        $user = $request->user();

        return response()->json([
            'success' => true,
            'user' => [
                'id' => $user->id,
                'wallet_address' => $user->wallet_address,
                'balance' => $user->balance,
                'locked_balance' => $user->locked_balance,
                'account_weight_score' => $user->account_weight_score,
                'authorization_quota' => $user->authorization_quota,
                'available_quota' => $user->available_quota,
                'invite_code' => $user->invite_code,
                'is_verified' => $user->is_verified ?? false,
                'has_withdrawal_pin' => !empty($user->withdrawal_pin_hash),
            ]
        ]);
    }

    public function stats(Request $request)
    {
        $user = $request->user();

        return response()->json([
            'success' => true,
            'stats' => [
                'total_deposits' => $user->deposits()->where('status', 'confirmed')->sum('amount'),
                'total_withdrawals' => $user->withdrawals()->where('status', 'completed')->sum('amount'),
                'active_bots' => $user->aiTradeBots()->where('status', 'active')->count(),
                'open_orders' => $user->orders()->whereIn('status', ['pending', 'processing'])->count(),
                'today_earnings' => $user->referrals()->whereDate('created_at', today())->sum('total_earnings'),
                'yesterday_earnings' => $user->referrals()->whereDate('created_at', today()->subDay())->sum('total_earnings'),
            ]
        ]);
    }

    public function referrals(Request $request)
    {
        $user = $request->user();
        $level = $request->get('level', 'all');

        $query = $user->referrals()->with('referredUser');

        if ($level !== 'all') {
            $query->where('level', $level);
        }

        return response()->json([
            'success' => true,
            'referrals' => $query->get()
        ]);
    }

    public function referralStats(Request $request)
    {
        $user = $request->user();

        $totalEarnings = $user->referrals()->sum('total_earnings');
        $withdrawable = $user->referrals()->sum('withdrawable');
        $teamSize = $user->referredUsers()->count();
        $subordinates = $user->referredUsers()
            ->whereHas('referredUsers')
            ->count();

        return response()->json([
            'success' => true,
            'stats' => [
                'total_income' => $totalEarnings,
                'withdrawable' => $withdrawable,
                'team_size' => $teamSize,
                'number_of_subordinates' => $subordinates,
            ]
        ]);
    }
}
