<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Swap;
use App\Models\UserBalance;
use App\Services\BinanceService;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SwapController extends Controller
{
    protected $binanceService;

    public function __construct(BinanceService $binanceService)
    {
        $this->binanceService = $binanceService;
    }

    public function index(Request $request)
    {
        $user = $request->user();
        $swaps = $user->swaps()->latest()->paginate(20);

        return response()->json([
            'success' => true,
            'swaps' => $swaps
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'from_coin' => 'required|string',
            'from_amount' => 'required|numeric|min:0.00000001',
            'to_coin' => 'required|string|different:from_coin',
            'from_price' => 'sometimes|numeric|min:0',
            'to_price' => 'sometimes|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = $request->user();
        $fromCoin = strtoupper($request->from_coin);
        $toCoin = strtoupper($request->to_coin);
        $fromAmount = (float) $request->from_amount;

        // Get user balance for the source coin
        $fromBalance = UserBalance::where('user_id', $user->id)
            ->where('coin_name', $fromCoin)
            ->first();

        if (!$fromBalance) {
            return response()->json([
                'success' => false,
                'message' => 'No balance found for ' . $fromCoin
            ], 400);
        }

        // Check available balance
        $availableBalance = (float) $fromBalance->balance - (float) $fromBalance->locked_balance;
        if ($fromAmount > $availableBalance) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance. Available: ' . number_format($availableBalance, 8) . ' ' . $fromCoin
            ], 400);
        }

        // Prefer server prices; when production cannot reach Binance, use client-provided prices
        $fromPrice = $this->binanceService->getCoinPrice($fromCoin);
        $toPrice = $this->binanceService->getCoinPrice($toCoin);
        if ($fromPrice === null || $toPrice === null) {
            $clientFrom = $request->has('from_price') ? (float) $request->from_price : null;
            $clientTo = $request->has('to_price') ? (float) $request->to_price : null;
            if ($clientFrom !== null && $clientTo !== null && $clientFrom > 0 && $clientTo > 0) {
                $fromPrice = $clientFrom;
                $toPrice = $clientTo;
                Log::info('Swap using client-provided prices (server market unreachable)', [
                    'from_coin' => $fromCoin, 'to_coin' => $toCoin,
                    'from_price' => $fromPrice, 'to_price' => $toPrice,
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Unable to fetch market prices. Please try again or refresh the page.'
                ], 500);
            }
        }

        try {
            // Calculate exchange rate and output amount
            $exchangeRate = $fromPrice / $toPrice;
            $swapFeePercent = 0.001;
            $feeAmount = $fromAmount * $swapFeePercent;
            $amountAfterFee = $fromAmount - $feeAmount;
            $toAmount = $amountAfterFee * $exchangeRate;

            $swap = DB::transaction(function () use ($user, $fromBalance, $fromCoin, $toCoin, $fromAmount, $toAmount, $exchangeRate, $feeAmount, $fromPrice, $toPrice) {
                $fromBalance->decrement('balance', $fromAmount);

                $toBalance = UserBalance::firstOrNew([
                    'user_id' => $user->id,
                    'coin_name' => $toCoin
                ]);

                if ($toBalance->exists) {
                    $toBalance->increment('balance', $toAmount);
                } else {
                    $toBalance->balance = $toAmount;
                    $toBalance->locked_balance = 0;
                    $toBalance->save();
                }

                if ($fromPrice && $toPrice) {
                    $fromUSD = $fromAmount * $fromPrice;
                    $toUSD = $toAmount * $toPrice;
                    $usdDifference = $toUSD - $fromUSD;
                    if ($usdDifference != 0) {
                        $user->increment('balance', $usdDifference);
                    }
                }

                return Swap::create([
                    'user_id' => $user->id,
                    'from_coin' => $fromCoin,
                    'from_amount' => $fromAmount,
                    'to_coin' => $toCoin,
                    'to_amount' => $toAmount,
                    'exchange_rate' => $exchangeRate,
                    'fee' => $feeAmount,
                    'status' => 'completed',
                ]);
            });

            return response()->json([
                'success' => true,
                'swap' => $swap,
                'message' => 'Swap completed successfully'
            ], 201);
        } catch (\Exception $e) {
            Log::error('Swap failed: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'from_coin' => $fromCoin,
                'to_coin' => $toCoin,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Swap could not be completed. Please try again.',
            ], 500);
        }
    }
}
