<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AiTradeBot;
use App\Models\Order;
use App\Models\UserBalance;
use App\Services\OrderService;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class AiTradeController extends Controller
{
    /**
     * List user's bots. Also runs expiry check so bots past admin_set_trading_time_minutes
     * are auto-closed even if cron didn't run (e.g. production scheduler not set).
     */
    public function index(Request $request, OrderService $orderService)
    {
        $user = $request->user();
        $bots = $user->aiTradeBots()->latest()->get();

        // Run expiry check for each active bot so auto-close happens when user opens app (no cron dependency)
        foreach ($bots as $bot) {
            if ($bot->status !== 'active') {
                continue;
            }
            if (!$bot->admin_activated_at) {
                continue;
            }
            $elapsed = (int) $bot->admin_activated_at->diffInMinutes(now(), true);
            $total = (int) ($bot->admin_set_trading_time_minutes ?? 1440); // Default 24 hours
            if ($elapsed >= $total) {
                try {
                    $orderService->processBot($bot->fresh());
                } catch (\Throwable $e) {
                    Log::error('Bot expiry check failed on index', ['bot_id' => $bot->id, 'error' => $e->getMessage()]);
                }
            }
        }

        $bots = $user->aiTradeBots()->latest()->get();

        return response()->json([
            'success' => true,
            'bots' => $bots
        ]);
    }

    public function store(Request $request)
    {
        // Log incoming request for debugging
        Log::info('AI Trade Bot Creation Request', [
            'user_id' => $request->user()?->id,
            'request_data' => $request->all()
        ]);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'trading_pair' => 'required|string',
            'leverage' => 'required|numeric|min:1|max:100',
            'amount' => 'required|numeric|min:500|max:999999999',
            'weight_range_min' => 'required|numeric|min:0|max:10',
            'weight_range_max' => 'required|numeric|min:0|max:10',
            'quota_range_min' => 'required|numeric|min:500',
            'quota_range_max' => 'required|numeric|min:500|max:999999999',
            'max_profit_percent' => 'required|numeric|min:0|max:100',
            'max_loss_percent' => 'required|numeric|min:0|max:100',
            'expected_return_min' => 'required|numeric|min:0',
            'expected_return_max' => 'required|numeric|max:100',
        ]);

        if ($validator->fails()) {
            $errors = $validator->errors();
            $firstError = $errors->first();
            Log::warning('AI Trade Bot Creation Validation Failed', [
                'errors' => $errors->toArray(),
                'request_data' => $request->all()
            ]);
            return response()->json([
                'success' => false,
                'message' => $firstError ?: 'Validation failed',
                'errors' => $errors
            ], 422);
        }

        $user = $request->user();
        
        if (!$user) {
            Log::error('AI Trade Bot Creation: User not authenticated');
            return response()->json([
                'success' => false,
                'message' => 'User not authenticated'
            ], 401);
        }

        // AI trading uses balance only (no quota). Min 500 USDT.
        $requestAmount = (float) $request->amount;

        Log::info('AI Trade Bot Creation: Balance Check', [
            'user_id' => $user->id,
            'request_amount' => $requestAmount,
        ]);

        // Get or create USDT balance from UserBalance table
        // Use fresh() to ensure we get the latest data from database
        $usdtBalance = UserBalance::where('user_id', $user->id)
            ->where('coin_name', 'USDT')
            ->first();

        // If UserBalance doesn't exist but user has legacy balance, migrate it now (before balance check)
        if (!$usdtBalance && ((float) $user->balance > 0 || (float) $user->locked_balance > 0)) {
            $initialBalance = max(0, (float) $user->balance);
            $initialLockedBalance = max(0, (float) $user->locked_balance);
            $usdtBalance = UserBalance::create([
                'user_id' => $user->id,
                'coin_name' => 'USDT',
                'balance' => $initialBalance,
                'locked_balance' => $initialLockedBalance,
            ]);
            Log::info('AI Trade Bot Creation: Auto-migrated Legacy Balance to UserBalance', [
                'user_id' => $user->id,
                'initial_balance' => $initialBalance,
                'initial_locked_balance' => $initialLockedBalance,
            ]);
        }

        // Refresh UserBalance to get latest data (in case it was updated elsewhere)
        if ($usdtBalance) {
            $usdtBalance = $usdtBalance->fresh();
        }

        // Sync locked balance with active bots (fix any discrepancies)
        $activeBots = $user->aiTradeBots()->whereIn('status', ['active', 'paused'])->get();
        $expectedLockedBalance = max(0, $activeBots->sum('amount'));
        
        if ($usdtBalance) {
            $currentLockedBalance = (float) $usdtBalance->locked_balance;
            if ($currentLockedBalance != $expectedLockedBalance) {
                // Sync locked balance with active bots
                $usdtBalance->update(['locked_balance' => $expectedLockedBalance]);
                $user->update(['locked_balance' => $expectedLockedBalance]);
                Log::info('AI Trade Bot Creation: Synced locked balance', [
                    'user_id' => $user->id,
                    'old_locked' => $currentLockedBalance,
                    'new_locked' => $expectedLockedBalance,
                    'active_bots_count' => $activeBots->count(),
                ]);
            }
        }

        // Calculate available USDT balance
        // If UserBalance exists, use it; otherwise fallback to legacy balance field
        $availableUSDTBalance = 0;
        if ($usdtBalance) {
            // Refresh again after sync
            $usdtBalance = $usdtBalance->fresh();
            $availableUSDTBalance = (float) $usdtBalance->balance - (float) $usdtBalance->locked_balance;
            Log::info('AI Trade Bot Creation: Using UserBalance', [
                'usdt_balance' => $usdtBalance->balance,
                'usdt_locked_balance' => $usdtBalance->locked_balance,
                'available' => $availableUSDTBalance,
            ]);
        } else {
            // Fallback: check legacy balance field (for backward compatibility)
            // If user has balance in legacy field but no UserBalance record, use legacy balance
            $availableUSDTBalance = max(0, (float) $user->balance - (float) $user->locked_balance);
            Log::info('AI Trade Bot Creation: Using Legacy Balance', [
                'legacy_balance' => $user->balance,
                'legacy_locked_balance' => $user->locked_balance,
                'available' => $availableUSDTBalance,
            ]);
        }

        // Check available USDT balance
        if ($requestAmount > $availableUSDTBalance) {
            Log::warning('AI Trade Bot Creation: Insufficient USDT Balance', [
                'user_id' => $user->id,
                'request_amount' => $requestAmount,
                'available_usdt_balance' => $availableUSDTBalance,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Insufficient USDT balance. Available: ' . number_format($availableUSDTBalance, 2) . ' USDT, Required: ' . number_format($requestAmount, 2) . ' USDT'
            ], 400);
        }

        // Create bot and lock balance in transaction
        try {
            $bot = \Illuminate\Support\Facades\DB::transaction(function () use ($user, $request, $usdtBalance, $requestAmount) {
                // Refresh USDT balance (it may have been created before transaction)
                $usdtBalance = UserBalance::where('user_id', $user->id)
                    ->where('coin_name', 'USDT')
                    ->first();
                
                // Get or create USDT balance (should already exist if legacy balance was migrated)
                if (!$usdtBalance) {
                    // If UserBalance doesn't exist but user has legacy balance, migrate it
                    $initialBalance = max(0, (float) $user->balance);
                    $initialLockedBalance = max(0, (float) $user->locked_balance);
                    $usdtBalance = UserBalance::create([
                        'user_id' => $user->id,
                        'coin_name' => 'USDT',
                        'balance' => $initialBalance,
                        'locked_balance' => $initialLockedBalance,
                    ]);
                    Log::info('AI Trade Bot Creation: Created UserBalance in Transaction', [
                        'user_id' => $user->id,
                        'initial_balance' => $initialBalance,
                        'initial_locked_balance' => $initialLockedBalance,
                    ]);
                }

                // Lock the USDT balance for this bot
                $usdtBalance->increment('locked_balance', $requestAmount);
                
                // Also update legacy balance fields for backward compatibility
                $user->increment('locked_balance', $requestAmount);

                $bot = AiTradeBot::create([
                    'user_id' => $user->id,
                    'name' => $request->name,
                    'trading_pair' => $request->trading_pair,
                    'leverage' => $request->leverage,
                    'amount' => $requestAmount,
                    'weight_range_min' => $request->weight_range_min,
                    'weight_range_max' => $request->weight_range_max,
                    'quota_range_min' => $request->quota_range_min,
                    'quota_range_max' => $request->quota_range_max,
                    'max_profit_percent' => $request->max_profit_percent,
                    'max_loss_percent' => $request->max_loss_percent,
                    'expected_return_min' => $request->expected_return_min,
                    'expected_return_max' => $request->expected_return_max,
                    'status' => 'active',
                ]);

                Log::info('AI Trade Bot Created Successfully', [
                    'bot_id' => $bot->id,
                    'user_id' => $user->id,
                    'amount' => $requestAmount,
                ]);

                return $bot;
            });

            return response()->json([
                'success' => true,
                'bot' => $bot,
                'message' => 'AI Trade bot created'
            ], 201);
        } catch (\Exception $e) {
            Log::error('AI Trade Bot Creation Error', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Failed to create bot: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $user = $request->user();
        $bot = $user->aiTradeBots()->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|string|max:255',
            'status' => 'sometimes|in:active,paused,stopped,closed',
            'max_profit_percent' => 'sometimes|numeric|min:0|max:100',
            'max_loss_percent' => 'sometimes|numeric|min:0|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $updateData = $request->only([
            'name', 'status', 'max_profit_percent', 'max_loss_percent'
        ]);

        // If status is being changed to closed, set closed_at
        if (isset($updateData['status']) && $updateData['status'] === 'closed') {
            $updateData['closed_at'] = now();
            if ($request->has('close_reason')) {
                $updateData['close_reason'] = $request->close_reason;
            }
        }

        $bot->update($updateData);

        return response()->json([
            'success' => true,
            'bot' => $bot,
            'message' => 'Bot updated'
        ]);
    }

    public function stop(Request $request, $id)
    {
        // Only admin can stop bots; users cannot stop from app or API
        return response()->json([
            'success' => false,
            'message' => 'Only admin can stop the bot. Please contact support.',
        ], 403);
    }

    public function close(Request $request, $id, OrderService $orderService)
    {
        // Only admin can close bots; users cannot close from app or API
        return response()->json([
            'success' => false,
            'message' => 'Only admin can close the bot. Please contact support.',
        ], 403);
    }
    
    public function destroy(Request $request, $id)
    {
        $user = $request->user();
        $bot = $user->aiTradeBots()->findOrFail($id);

        // Unlock balance and restore quota when bot is deleted
        \Illuminate\Support\Facades\DB::transaction(function () use ($user, $bot) {
            // Get USDT balance
            $usdtBalance = UserBalance::where('user_id', $user->id)
                ->where('coin_name', 'USDT')
                ->first();

            if ($usdtBalance) {
                // Unlock USDT balance - prevent negative
                $currentLocked = max(0, (float) $usdtBalance->locked_balance);
                $unlockAmount = min($currentLocked, $bot->amount);
                if ($unlockAmount > 0) {
                    $usdtBalance->decrement('locked_balance', $unlockAmount);
                }
            }

            // Also update legacy balance for backward compatibility - prevent negative
            $currentLocked = max(0, (float) $user->locked_balance);
            $unlockAmount = min($currentLocked, $bot->amount);
            if ($unlockAmount > 0) {
                $user->decrement('locked_balance', $unlockAmount);
            }

            // Delete the bot
            $bot->delete();
        });

        return response()->json([
            'success' => true,
            'message' => 'Bot deleted and balance unlocked'
        ]);
    }

    public function toggle(Request $request, $id)
    {
        $user = $request->user();
        $bot = $user->aiTradeBots()->findOrFail($id);

        $newStatus = $bot->status === 'active' ? 'paused' : 'active';
        $bot->update(['status' => $newStatus]);

        return response()->json([
            'success' => true,
            'bot' => $bot,
            'message' => "Bot {$newStatus}"
        ]);
    }

    public function orders(Request $request)
    {
        $user = $request->user();
        
        $query = $user->orders()->with('aiTradeBot');
        
        // Filter by status
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by trading pair
        if ($request->has('trading_pair')) {
            $query->where('trading_pair', $request->trading_pair);
        }
        
        // Filter by date range
        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }
        
        // Sort by executed_at (most recent first), fallback to created_at
        $orders = $query->orderBy('executed_at', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 20);
        
        // Get all unique bot IDs from orders to fetch bot summaries
        $botIds = $orders->getCollection()
            ->pluck('ai_trade_bot_id')
            ->filter()
            ->unique()
            ->toArray();
        
        // Fetch all bots with their final P/L calculated
        $bots = [];
        if (!empty($botIds)) {
            $botsCollection = \App\Models\AiTradeBot::whereIn('id', $botIds)->get();
            foreach ($botsCollection as $bot) {
                if ($bot->status === 'closed') {
                    $bots[$bot->id] = [
                        'bot_id' => $bot->id,
                        'bot_name' => $bot->name,
                        'bot_status' => $bot->status,
                        'bot_amount' => $bot->amount,
                        'total_profit' => (float) $bot->total_profit,
                        'total_loss' => (float) $bot->total_loss,
                        'net_pl' => (float) $bot->total_profit - (float) $bot->total_loss,
                        'closed_at' => $bot->closed_at,
                    ];
                } else {
                    $bots[$bot->id] = [
                        'bot_id' => $bot->id,
                        'bot_name' => $bot->name,
                        'bot_status' => $bot->status,
                        'bot_amount' => $bot->amount,
                        'total_profit' => (float) $bot->total_profit,
                        'total_loss' => (float) $bot->total_loss,
                        'net_pl' => (float) $bot->total_profit - (float) $bot->total_loss,
                    ];
                }
            }
        }
        
        // Add bot summary to each order (matching closed section results)
        $orders->getCollection()->transform(function($order) use ($bots) {
            if ($order->aiTradeBot && isset($bots[$order->aiTradeBot->id])) {
                $order->bot_summary = $bots[$order->aiTradeBot->id];
            }
            return $order;
        });

        return response()->json([
            'success' => true,
            'orders' => $orders
        ]);
    }

    public function stats(Request $request)
    {
        $user = $request->user();
        
        $activeBots = $user->aiTradeBots()->where('status', 'active')->count();
        $stoppedBots = $user->aiTradeBots()->where('status', 'stopped')->count();
        $closedBots = $user->aiTradeBots()->where('status', 'closed')->count();
        
        $totalProfit = $user->aiTradeBots()->sum('total_profit');
        $totalLoss = $user->aiTradeBots()->sum('total_loss');
        $netProfit = $totalProfit - $totalLoss;
        
        $totalOrders = $user->orders()->count();
        $filledOrders = $user->orders()->where('status', 'filled')->count();
        
        return response()->json([
            'success' => true,
            'stats' => [
                'active_bots' => $activeBots,
                'stopped_bots' => $stoppedBots,
                'closed_bots' => $closedBots,
                'total_profit' => $totalProfit,
                'total_loss' => $totalLoss,
                'net_profit' => $netProfit,
                'total_orders' => $totalOrders,
                'filled_orders' => $filledOrders,
            ]
        ]);
    }
}
