<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Wallet;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Storage;

class WalletManagementController extends Controller
{
    public function index(): View
    {
        $wallets = Wallet::latest()->get();

        return view('admin.wallets.index', [
            'wallets' => $wallets,
            'pageTitle' => __('admin.nav.wallets'),
            'title' => 'Wallet Management'
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'name' => 'nullable|string',
            'coin_name' => 'required|string|max:10',
            'network' => 'required|string',
            'address' => 'required|string',
            'qr_code' => 'nullable|image|max:2048',
            'is_active' => 'boolean',
        ]);

        // Auto-generate name if not provided
        if (empty($validated['name'])) {
            $validated['name'] = $validated['network'] . ' Wallet';
        }

        // Handle QR code upload
        if ($request->hasFile('qr_code')) {
            $validated['qr_code_path'] = $request->file('qr_code')->store('wallets/qr-codes', 'public');
        }

        unset($validated['qr_code']); // Remove file from validated array before creating

        Wallet::create($validated);

        \App\Models\AuditLog::create([
            'admin_user_id' => session('admin.user_id'),
            'action' => 'wallet_create',
            'model_type' => Wallet::class,
            'changes' => json_encode($validated),
            'reason' => 'Wallet management',
        ]);

        return redirect()->back()->with('success', 'Wallet created');
    }

    public function update(Request $request, $id): RedirectResponse
    {
        $wallet = Wallet::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|string',
            'coin_name' => 'sometimes|string|max:10',
            'network' => 'sometimes|string',
            'address' => 'sometimes|string',
            'qr_code' => 'nullable|image|max:2048',
            'is_active' => 'boolean',
        ]);

        // Handle QR code upload
        if ($request->hasFile('qr_code')) {
            // Delete old QR code if exists
            if ($wallet->qr_code_path && Storage::disk('public')->exists($wallet->qr_code_path)) {
                Storage::disk('public')->delete($wallet->qr_code_path);
            }
            $validated['qr_code_path'] = $request->file('qr_code')->store('wallets/qr-codes', 'public');
        }

        unset($validated['qr_code']); // Remove file from validated array

        $wallet->update($validated);

        \App\Models\AuditLog::create([
            'admin_user_id' => session('admin.user_id'),
            'action' => 'wallet_update',
            'model_type' => Wallet::class,
            'model_id' => $wallet->id,
            'changes' => json_encode($validated),
            'reason' => 'Wallet management',
        ]);

        return redirect()->back()->with('success', 'Wallet updated');
    }

    public function destroy($id): RedirectResponse
    {
        $wallet = Wallet::findOrFail($id);
        $wallet->delete();

        \App\Models\AuditLog::create([
            'admin_user_id' => session('admin.user_id'),
            'action' => 'wallet_delete',
            'model_type' => Wallet::class,
            'model_id' => $wallet->id,
            'reason' => 'Wallet management',
        ]);

        return redirect()->back()->with('success', 'Wallet deleted');
    }
}
