<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Notification;
use App\Services\NotificationService;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\DB;

class NotificationController extends Controller
{
    public function index(Request $request): View
    {
        $query = User::query();
        if ($request->has('search')) {
            $s = $request->get('search');
            $query->where(function ($q) use ($s) {
                $q->where('wallet_address', 'like', "%{$s}%")
                  ->orWhere('name', 'like', "%{$s}%")
                  ->orWhere('id', 'like', "%{$s}%");
            });
        }
        $users = $query->latest()->paginate(20)->withQueryString();

        $templateKeys = ['notification_welcome_title', 'notification_welcome_message', 'notification_deposit_approved_title', 'notification_deposit_approved_message', 'notification_withdraw_approved_title', 'notification_withdraw_approved_message'];
        $templates = [];
        foreach ($templateKeys as $key) {
            $templates[$key] = NotificationService::getTemplate($key);
        }

        return view('admin.notifications.index', [
            'users' => $users,
            'templates' => $templates,
            'pageTitle' => __('admin.nav.notifications'),
            'title' => 'Send Notifications & Templates',
        ]);
    }

    public function send(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'user_ids' => 'required|array',
            'user_ids.*' => 'required|exists:users,id',
            'message_type' => 'required|in:maintenance,success,info,custom',
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
        ]);

        $count = 0;
        foreach ($validated['user_ids'] as $userId) {
            $user = User::find($userId);
            if ($user) {
                NotificationService::createAdminMessage(
                    $user,
                    $validated['message_type'],
                    $validated['title'],
                    $validated['message']
                );
                $count++;
            }
        }

        return redirect()->back()->with('success', "Notification sent to {$count} user(s).");
    }

    public function updateTemplates(Request $request): RedirectResponse
    {
        $keys = [
            'notification_welcome_title', 'notification_welcome_message',
            'notification_deposit_approved_title', 'notification_deposit_approved_message',
            'notification_withdraw_approved_title', 'notification_withdraw_approved_message',
        ];
        foreach ($keys as $key) {
            $value = $request->input($key);
            if ($value !== null) {
                $exists = DB::table('system_settings')->where('key', $key)->exists();
                if ($exists) {
                    DB::table('system_settings')->where('key', $key)->update(['value' => $value, 'updated_at' => now()]);
                } else {
                    DB::table('system_settings')->insert(['key' => $key, 'value' => $value, 'description' => $key, 'created_at' => now(), 'updated_at' => now()]);
                }
            }
        }
        return redirect()->back()->with('success', 'Notification templates updated.');
    }

}
