<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use App\Models\DiscoverPost;
use App\Models\DiscoverPostComment;
use App\Models\DiscoverPostLike;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class DiscoverController extends Controller
{
    /** Random 6-digit demo author ID in 100000–599999 (real user IDs start from 600000, so no clash). */
    private function randomDemoAuthorId(): string
    {
        return (string) random_int(100000, 599999);
    }

    public function index(Request $request): View
    {
        $tab = $request->get('tab', 'demo');
        if ($tab === 'users') {
            $posts = DiscoverPost::withCount(['likes', 'comments'])->whereNotNull('user_id')->latest()->paginate(20)->withQueryString();
        } else {
            $tab = 'demo';
            $posts = DiscoverPost::withCount(['likes', 'comments'])->whereNull('user_id')->latest()->paginate(20)->withQueryString();
        }
        $demoAuthorIds = DiscoverPost::whereNull('user_id')->distinct()->pluck('demo_author_id')->filter(function ($v) {
            return is_numeric($v) && (int) $v >= 100000 && (int) $v <= 599999;
        })->map(fn ($v) => (int) $v)->sort()->values()->unique()->all();
        return view('admin.discover.index', [
            'posts' => $posts,
            'demoAuthorIds' => $demoAuthorIds,
            'tab' => $tab,
            'pageTitle' => $tab === 'users' ? __('admin.common.users_discover_tab') : __('admin.common.discover_posts_tab'),
            'title' => 'Discover Management',
        ]);
    }

    public function create(): View
    {
        return view('admin.discover.create', [
            'pageTitle' => 'Add Demo Post',
            'title' => 'Add Discover Post',
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'content' => 'required|string|max:5000',
            'image' => 'nullable|image|max:5120',
            'demo_author_id' => 'required|integer|min:100000|max:599999',
        ], [
            'demo_author_id.min' => 'Demo Author ID must be 6 digits (100000–599999). Real user IDs start from 600000.',
            'demo_author_id.max' => 'Demo Author ID must be below 600000 to avoid clashing with real users.',
        ]);
        $demoId = (string) $validated['demo_author_id'];
        if (DiscoverPost::whereNull('user_id')->where('demo_author_id', $demoId)->exists()) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['demo_author_id' => 'This Demo Author ID is already used by another demo post. Please choose a different 6-digit ID (100000–599999).']);
        }
        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('discover/posts', 'public');
        }
        DiscoverPost::create([
            'user_id' => null,
            'demo_author_id' => $demoId,
            'content' => $validated['content'],
            'image_path' => $imagePath,
        ]);
        return redirect()->route('admin.discover')->with('success', 'Demo post created (Author: User ' . $demoId . ')');
    }

    public function edit($id): View
    {
        $post = DiscoverPost::findOrFail($id);
        return view('admin.discover.edit', [
            'post' => $post,
            'pageTitle' => 'Edit Discover Post',
            'title' => 'Edit Post',
        ]);
    }

    public function update(Request $request, $id): RedirectResponse
    {
        $post = DiscoverPost::findOrFail($id);
        $validated = $request->validate([
            'content' => 'required|string|max:5000',
            'image' => 'nullable|image|max:5120',
        ]);
        $imagePath = $post->image_path;
        if ($request->hasFile('image')) {
            if ($post->image_path && Storage::disk('public')->exists($post->image_path)) {
                Storage::disk('public')->delete($post->image_path);
            }
            $imagePath = $request->file('image')->store('discover/posts', 'public');
        }
        $post->update(['content' => $validated['content'], 'image_path' => $imagePath]);
        return redirect()->route('admin.discover')->with('success', 'Post updated');
    }

    public function destroy(Request $request, $id): RedirectResponse
    {
        $post = DiscoverPost::findOrFail($id);
        if ($post->image_path && Storage::disk('public')->exists($post->image_path)) {
            Storage::disk('public')->delete($post->image_path);
        }
        $post->delete();
        $tab = $request->input('_redirect_tab', 'demo');
        return redirect()->route('admin.discover', ['tab' => $tab])->with('success', 'Post deleted');
    }

    public function storeComment(Request $request, $id): RedirectResponse
    {
        $validated = $request->validate([
            'body' => 'required|string|max:2000',
            'demo_author_id' => 'required|integer|min:100000|max:599999',
        ], [
            'demo_author_id.min' => 'Choose a demo author ID (100000–599999).',
            'demo_author_id.max' => 'Demo author ID must be below 600000.',
        ]);
        $post = DiscoverPost::findOrFail($id);
        $authorDisplay = (string) $validated['demo_author_id'];
        DiscoverPostComment::create([
            'discover_post_id' => $post->id,
            'user_id' => null,
            'author_display' => $authorDisplay,
            'body' => $validated['body'],
        ]);
        return redirect()->back()->with('success', 'Comment added (User ' . $authorDisplay . ')');
    }

    /** Add demo likes (no user — frontend does not show who liked). */
    public function storeLikes(Request $request, $id): RedirectResponse
    {
        $validated = $request->validate([
            'count' => 'required|integer|min:1|max:500',
        ], ['count.min' => 'Add at least 1 like.', 'count.max' => 'Maximum 500 likes at once.']);
        $post = DiscoverPost::findOrFail($id);
        $count = (int) $validated['count'];
        for ($i = 0; $i < $count; $i++) {
            DiscoverPostLike::create([
                'discover_post_id' => $post->id,
                'user_id' => null,
            ]);
        }
        return redirect()->back()->with('success', $count . ' like(s) added.');
    }
}
