<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AdminUser;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Hash;

class AdminManagementController extends Controller
{
    public function index(): View
    {
        $admins = AdminUser::latest()->get();

        return view('admin.admins.index', [
            'admins' => $admins,
            'pageTitle' => __('admin.nav.admins'),
            'title' => 'Admin Management'
        ]);
    }

    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'username' => 'required|string|unique:admin_users',
            'email' => 'required|email|unique:admin_users',
            'password' => 'required|string|min:8',
            'role' => 'required|in:super_admin,admin,service_admin',
        ]);

        $validated['password'] = Hash::make($validated['password']);
        $admin = AdminUser::create($validated);

        \App\Models\AuditLog::create([
            'admin_user_id' => session('admin.user_id'),
            'action' => 'admin_create',
            'model_type' => AdminUser::class,
            'model_id' => $admin->id,
            'changes' => json_encode(['username' => $admin->username, 'role' => $admin->role]),
            'reason' => 'Admin management',
        ]);

        return redirect()->back()->with('success', 'Admin created');
    }

    public function update(Request $request, $id): RedirectResponse
    {
        $admin = AdminUser::findOrFail($id);

        $validated = $request->validate([
            'email' => 'sometimes|email|unique:admin_users,email,' . $id,
            'password' => 'sometimes|string|min:8',
            'role' => 'sometimes|in:super_admin,admin,service_admin',
            'is_active' => 'boolean',
        ]);

        if (isset($validated['password'])) {
            $validated['password'] = Hash::make($validated['password']);
        }

        $admin->update($validated);

        \App\Models\AuditLog::create([
            'admin_user_id' => session('admin.user_id'),
            'action' => 'admin_update',
            'model_type' => AdminUser::class,
            'model_id' => $admin->id,
            'changes' => json_encode($validated),
            'reason' => 'Admin management',
        ]);

        return redirect()->back()->with('success', 'Admin updated');
    }

    public function destroy($id): RedirectResponse
    {
        $admin = AdminUser::findOrFail($id);

        if ($admin->role === 'super_admin') {
            return redirect()->back()->with('error', __('admin.admins.cannot_delete_super_admin'));
        }

        $admin->delete();

        \App\Models\AuditLog::create([
            'admin_user_id' => session('admin.user_id'),
            'action' => 'admin_delete',
            'model_type' => AdminUser::class,
            'model_id' => $admin->id,
            'reason' => 'Admin management',
        ]);

        return redirect()->back()->with('success', 'Admin deleted');
    }
}
